/*
 * Copyright (c) 1998, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.javadoc.internal.doclets.toolkit.util;

import java.io.*;
import java.net.*;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import javax.lang.model.element.Element;
import javax.lang.model.element.ModuleElement;
import javax.lang.model.element.PackageElement;
import javax.tools.Diagnostic;
import javax.tools.DocumentationTool;

import com.sun.tools.javac.code.Flags;
import com.sun.tools.javac.code.Symbol.ModuleSymbol;
import jdk.javadoc.doclet.Reporter;
import jdk.javadoc.internal.doclets.toolkit.BaseConfiguration;

/**
 * Process and manage "-link" and "-linkoffline" to external packages. The
 * options "-link" and "-linkoffline" both depend on the fact that Javadoc now
 * generates "package-list"(lists all the packages which are getting
 * documented) file in the current or the destination directory, while
 * generating the documentation.
 *
 *  <p><b>This is NOT part of any supported API.
 *  If you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 *
 * @author Atul M Dambalkar
 * @author Robert Field
 */
public class Extern {

    /**
     * Map element names onto Extern Item objects.
     * Lazily initialized.
     */
    private Map<String, Item> moduleItems = new HashMap<>();
    private Map<String, Map<String, Item>> packageItems = new HashMap<>();

    /**
     * The global configuration information for this run.
     */
    private final BaseConfiguration configuration;

    /**
     * True if we are using -linkoffline and false if -link is used instead.
     */
    private boolean linkoffline = false;

    /**
     * Stores the info for one external doc set
     */
    private static class Item {

        /**
         * Element name, found in the "element-list" file in the {@link path}.
         */
        final String elementName;

        /**
         * The URL or the directory path at which the element documentation will be
         * avaliable.
         */
        final DocPath path;

        /**
         * If given path is directory path then true else if it is a URL then false.
         */
        final boolean relative;

        /**
         * Indicates that docs use old-form of anchors.
         */
        final boolean useOldFormId;

        /**
         * Constructor to build a Extern Item object and map it with the element name.
         * If the same element name is found in the map, then the first mapped
         * Item object or offline location will be retained.
         *
         * @param elementName Element name found in the "element-list" file.
         * @param path        URL or Directory path from where the "element-list"
         * file is picked.
         * @param relative    True if path is URL, false if directory path.
         */
        Item(String elementName, DocPath path, boolean relative, boolean useOldFormId) {
            this.elementName = elementName;
            this.path = path;
            this.relative = relative;
            this.useOldFormId = useOldFormId;
        }

        /**
         * String representation of "this" with elementname and the path.
         */
        @Override
        public String toString() {
            return elementName + (relative? " -> " : " => ") + path.getPath();
        }
    }

    public Extern(BaseConfiguration configuration) {
        this.configuration = configuration;
    }

    /**
     * Determine if a element item is externally documented.
     *
     * @param element an Element.
     * @return true if the element is externally documented
     */
    public boolean isExternal(Element element) {
        if (packageItems.isEmpty()) {
            return false;
        }
        PackageElement pe = configuration.utils.containingPackage(element);
        if (pe.isUnnamed()) {
            return false;
        }

        return findElementItem(pe) != null;
    }

    /**
     * Determine if a element item is a module or not.
     *
     * @param elementName name of the element.
     * @return true if the element is a module
     */
    public boolean isModule(String elementName) {
        Item elem = moduleItems.get(elementName);
        return (elem == null) ? false : true;
    }

    /**
     * Convert a link to be an external link if appropriate.
     *
     * @param element The element .
     * @param relativepath    The relative path.
     * @param filename    The link to convert.
     * @return if external return converted link else return null
     */
    public DocLink getExternalLink(Element element, DocPath relativepath, String filename) {
        return getExternalLink(element, relativepath, filename, null);
    }

    public DocLink getExternalLink(Element element, DocPath relativepath, String filename,
            String memberName) {
        Item fnd = findElementItem(element);
        if (fnd == null)
            return null;

        // The following is somewhat questionable since we are using DocPath
        // to contain external URLs!
        DocPath p = fnd.relative ?
                relativepath.resolve(fnd.path).resolve(filename) :
                fnd.path.resolve(filename);
        return new DocLink(p, "is-external=true", fnd.useOldFormId ? getOldFormHtmlName(memberName) : memberName);
    }

    /**
     * Build the extern element list from given URL or the directory path,
     * as specified with the "-link" flag.
     * Flag error if the "-link" or "-linkoffline" option is already used.
     *
     * @param url        URL or Directory path.
     * @param reporter   The <code>DocErrorReporter</code> used to report errors.
     * @return true if successful, false otherwise
     * @throws DocFileIOException if there is a problem reading a element list file
     */
    public boolean link(String url, Reporter reporter) throws DocFileIOException {
        return link(url, url, reporter, false);
    }

    /**
     * Build the extern element list from given URL or the directory path,
     * as specified with the "-linkoffline" flag.
     * Flag error if the "-link" or "-linkoffline" option is already used.
     *
     * @param url        URL or Directory path.
     * @param elemlisturl This can be another URL for "element-list" or ordinary
     *                   file.
     * @param reporter   The <code>DocErrorReporter</code> used to report errors.
     * @return true if successful, false otherwise
     * @throws DocFileIOException if there is a problem reading the element list file
     */
    public boolean link(String url, String elemlisturl, Reporter reporter) throws DocFileIOException {
        return link(url, elemlisturl, reporter, true);
    }

    /*
     * Build the extern element list from given URL or the directory path.
     * Flag error if the "-link" or "-linkoffline" option is already used.
     *
     * @param url        URL or Directory path.
     * @param elemlisturl This can be another URL for "element-list" or ordinary
     *                   file.
     * @param reporter   The <code>DocErrorReporter</code> used to report errors.
     * @param linkoffline True if -linkoffline is used and false if -link is used.
     * @return true if successful, false otherwise
     * @throws DocFileIOException if there is a problem reading the element list file
     */
    private boolean link(String url, String elemlisturl, Reporter reporter, boolean linkoffline)
                throws DocFileIOException {
        this.linkoffline = linkoffline;
        try {
            url = adjustEndFileSeparator(url);
            if (isUrl(elemlisturl)) {
                readElementListFromURL(url, toURL(adjustEndFileSeparator(elemlisturl)));
            } else {
                readElementListFromFile(url, DocFile.createFileForInput(configuration, elemlisturl));
            }
            return true;
        } catch (Fault f) {
            reporter.print(Diagnostic.Kind.ERROR, f.getMessage());
            return false;
        }
    }

    private URL toURL(String url) throws Fault {
        try {
            return new URL(url);
        } catch (MalformedURLException e) {
            throw new Fault(configuration.getText("doclet.MalformedURL", url), e);
        }
    }

    private static class Fault extends Exception {
        private static final long serialVersionUID = 0;

        Fault(String msg, Exception cause) {
            super(msg, cause);
        }
    }

    /**
     * Get the Extern Item object associated with this element name.
     *
     * @param element Element
     */
    private Item findElementItem(Element element) {
        Item item = null;
        if (element instanceof ModuleElement) {
            item = moduleItems.get(configuration.utils.getModuleName((ModuleElement)element));
        }
        else if (element instanceof PackageElement) {
            PackageElement packageElement = (PackageElement)element;
            ModuleElement moduleElement = configuration.utils.containingModule(packageElement);
            Map<String, Item> pkgMap = packageItems.get(configuration.utils.getModuleName(moduleElement));
            item = (pkgMap != null) ? pkgMap.get(configuration.utils.getPackageName(packageElement)) : null;
        }
        return item;
    }

    /**
     * If the URL or Directory path is missing end file separator, add that.
     */
    private String adjustEndFileSeparator(String url) {
        return url.endsWith("/") ? url : url + '/';
    }

    /**
     * Fetch the URL and read the "element-list" file.
     *
     * @param urlpath        Path to the elements.
     * @param elemlisturlpath URL or the path to the "element-list" file.
     */
    private void readElementListFromURL(String urlpath, URL elemlisturlpath) throws Fault {
        try {
            URL link = elemlisturlpath.toURI().resolve(DocPaths.ELEMENT_LIST.getPath()).toURL();
            readElementList(link.openStream(), urlpath, false, false);
        } catch (URISyntaxException | MalformedURLException exc) {
            throw new Fault(configuration.getText("doclet.MalformedURL", elemlisturlpath.toString()), exc);
        } catch (IOException exc) {
            readAlternateURL(urlpath, elemlisturlpath);
        }
    }

    /**
     * Fetch the URL and read the "package-list" file.
     *
     * @param urlpath        Path to the packages.
     * @param elemlisturlpath URL or the path to the "package-list" file.
     */
    private void readAlternateURL(String urlpath, URL elemlisturlpath) throws Fault {
        try {
            URL link = elemlisturlpath.toURI().resolve(DocPaths.PACKAGE_LIST.getPath()).toURL();
            readElementList(link.openStream(), urlpath, false, true);
        } catch (URISyntaxException | MalformedURLException exc) {
            throw new Fault(configuration.getText("doclet.MalformedURL", elemlisturlpath.toString()), exc);
        } catch (IOException exc) {
            throw new Fault(configuration.getText("doclet.URL_error", elemlisturlpath.toString()), exc);
        }
    }

    /**
     * Read the "element-list" file which is available locally.
     *
     * @param path URL or directory path to the elements.
     * @param elemListPath Path to the local "element-list" file.
     * @throws Fault if an error occurs that can be treated as a warning
     * @throws DocFileIOException if there is a problem opening the element list file
     */
    private void readElementListFromFile(String path, DocFile elemListPath)
            throws Fault, DocFileIOException {
        DocFile file = elemListPath.resolve(DocPaths.ELEMENT_LIST);
        if (! (file.isAbsolute() || linkoffline)){
            file = file.resolveAgainst(DocumentationTool.Location.DOCUMENTATION_OUTPUT);
        }
        if (file.exists()) {
            readElementList(file, path, false);
        } else {
            DocFile file1 = elemListPath.resolve(DocPaths.PACKAGE_LIST);
            if (!(file1.isAbsolute() || linkoffline)) {
                file1 = file1.resolveAgainst(DocumentationTool.Location.DOCUMENTATION_OUTPUT);
            }
            if (file1.exists()) {
                readElementList(file1, path, true);
            } else {
                throw new Fault(configuration.getText("doclet.File_error", file.getPath()), null);
            }
        }
    }

    private void readElementList(DocFile file, String path, boolean isOldFormDoc) throws Fault, DocFileIOException {
        try {
            if (file.canRead()) {
                boolean pathIsRelative
                        = !isUrl(path)
                        && !DocFile.createFileForInput(configuration, path).isAbsolute();
                readElementList(file.openInputStream(), path, pathIsRelative, isOldFormDoc);
            } else {
                throw new Fault(configuration.getText("doclet.File_error", file.getPath()), null);
            }
        } catch (IOException exc) {
           throw new Fault(configuration.getText("doclet.File_error", file.getPath()), exc);
        }
    }

    /**
     * Read the file "element-list" and for each element name found, create
     * Extern object and associate it with the element name in the map.
     *
     * @param input     InputStream from the "element-list" file.
     * @param path     URL or the directory path to the elements.
     * @param relative Is path relative?
     * @throws IOException if there is a problem reading or closing the stream
     */
    private void readElementList(InputStream input, String path, boolean relative, boolean isOldFormDoc)
                         throws IOException {
        try (BufferedReader in = new BufferedReader(new InputStreamReader(input))) {
            String elemname = null;
            String moduleName = null;
            DocPath elempath = null;
            DocPath basePath  = DocPath.create(path);
            while ((elemname = in.readLine()) != null) {
                if (elemname.length() > 0) {
                    elempath = basePath;
                    if (elemname.startsWith(DocletConstants.MODULE_PREFIX)) {
                        moduleName = elemname.replace(DocletConstants.MODULE_PREFIX, "");
                        Item item = new Item(moduleName, elempath, relative, isOldFormDoc);
                        moduleItems.put(moduleName, item);
                    } else {
                        DocPath pkgPath = DocPath.create(elemname.replace('.', '/'));
                        if (configuration.useModuleDirectories && moduleName != null) {
                            elempath = elempath.resolve(DocPath.create(moduleName).resolve(pkgPath));
                        } else {
                            elempath = elempath.resolve(pkgPath);
                        }
                        String actualModuleName = checkLinkCompatibility(elemname, moduleName, path);
                        Item item = new Item(elemname, elempath, relative, isOldFormDoc);
                        packageItems.computeIfAbsent(actualModuleName, k -> new TreeMap<>())
                            .put(elemname, item);
                    }
                }
            }
        }
    }

    // The following should be replaced by a new method such as Elements.isAutomaticModule
    private boolean isAutomaticModule(ModuleElement me) {
        if (me == null) {
            return false;
        } else {
            ModuleSymbol msym = (ModuleSymbol) me;
            return (msym.flags() & Flags.AUTOMATIC_MODULE) != 0;
        }
    }

    public boolean isUrl (String urlCandidate) {
        try {
            URL ignore = new URL(urlCandidate);
            //No exception was thrown, so this must really be a URL.
            return true;
        } catch (MalformedURLException e) {
            //Since exception is thrown, this must be a directory path.
            return false;
        }
    }

    /**
     * Check if the external documentation format matches our internal model of the code.
     * Returns the module name to use for external reference lookup according to the actual
     * modularity of the external package (and regardless of modularity of documentation).
     *
     * @param packageName the package name
     * @param moduleName the module name or null
     * @param path the documentation path
     * @return the module name to use according to actual modularity of the package
     */
    private String checkLinkCompatibility(String packageName, String moduleName, String path)  {
        PackageElement pe = configuration.utils.elementUtils.getPackageElement(packageName);
        if (pe != null) {
            ModuleElement me = (ModuleElement)pe.getEnclosingElement();
            if (me == null || me.isUnnamed()) {
                if (moduleName != null) {
                    configuration.getReporter().print(Diagnostic.Kind.WARNING,
                            configuration.getResources().getText("doclet.linkMismatch_PackagedLinkedtoModule", path));
                }
                // library is not modular, ignore module name even if documentation is modular
                return DocletConstants.DEFAULT_ELEMENT_NAME;
            } else if (moduleName == null) {
                // suppress the warning message in the case of automatic modules
                if (!isAutomaticModule(me)) {
                    configuration.getReporter().print(Diagnostic.Kind.WARNING,
                            configuration.getResources().getText("doclet.linkMismatch_ModuleLinkedtoPackage", path));
                }
                // library is modular, use module name for lookup even though documentation is not
                return configuration.utils.getModuleName(me);
            }
        }
        return moduleName == null ? DocletConstants.DEFAULT_ELEMENT_NAME : moduleName;
    }

    /**
     * Converts a name to an old-form HTML name (old-form id).
     *
     * @param name the string that needs to be converted to a valid HTML name
     * @return old-form HTML name
     */
    public static String getOldFormHtmlName(String name) {
        /* The HTML 4 spec at http://www.w3.org/TR/html4/types.html#h-6.2 mentions
         * that the name/id should begin with a letter followed by other valid characters.
         * The HTML 5 spec (draft) is more permissive on names/ids where the only restriction
         * is that it should be at least one character long and should not contain spaces.
         * The spec draft is @ http://www.w3.org/html/wg/drafts/html/master/dom.html#the-id-attribute.
         *
         * For HTML 4, we need to check for non-characters at the beginning of the name and
         * substitute it accordingly, "_" and "$" can appear at the beginning of a member name.
         * The method substitutes "$" with "Z:Z:D" and will prefix "_" with "Z:Z".
         */

        if (null == name)
            return name;

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < name.length(); i++) {
            char ch = name.charAt(i);
            switch (ch) {
                case '(':
                case ')':
                case '<':
                case '>':
                case ',':
                    sb.append('-');
                    break;
                case ' ':
                case '[':
                    break;
                case ']':
                    sb.append(":A");
                    break;
                // Any appearance of $ needs to be substituted with ":D" and not with hyphen
                // since a field name "P$$ and a method P(), both valid member names, can end
                // up as "P--". A member name beginning with $ needs to be substituted with
                // "Z:Z:D".
                case '$':
                    if (i == 0)
                        sb.append("Z:Z");
                    sb.append(":D");
                    break;
                // A member name beginning with _ needs to be prefixed with "Z:Z" since valid anchor
                // names can only begin with a letter.
                case '_':
                    if (i == 0)
                        sb.append("Z:Z");
                    sb.append(ch);
                    break;
                default:
                    sb.append(ch);
            }
        }
        return sb.toString();
    }
}
